/** @file   animation.cpp
 * @brief   Implementation of Animation - class.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */
 
#include "animation.h"
#include <exception>
#include <allegro.h>
#include "datatypes.h"
#include "sound.h"
//#include "StreamUtils.h"
#include "eng2dPrivateConstants.h"
using std::vector;
using std::istream;
using std::string;

namespace eng2d {



// Animation element opening and closing tags
#define ANIMATION_OPENINGTAG  "<eng2d_animation>"
#define ANIMATION_CLOSINGTAG  "</eng2d_animation>"
#define ANIMATION_V2_OPENINGTAG "<eng2d_animation_v2>"
#define ANIMATION_V2_CLOSINGTAG "</eng2d_animation_v2>"
// Playmode element opening and closing tags
#define PLAYMODE_OPENINGTAG   "<playmode>"
#define PLAYMODE_CLOSINGTAG   "</playmode>"
// Tags that mark the general animation info section
#define HEADER_OPENINGTAG    "<header>"
#define HEADER_CLOSINGTAG    "</header>"
// Tags that mark the frames element
#define BODY_OPENINGTAG   "<body>"
#define BODY_CLOSINGTAG   "</body>"
// Tags that mark frame element
#define FRAME_OPENINGTAG  "<f>"
#define FRAME_CLOSINGTAG  "</f>"


//********************************************************************
//                                                                   *
//      Static members and constants                                 *
//                                                                   *
//********************************************************************

/** Places the bitmap-pointers to the frame-structure from given
 * bitmap-table
 */
void Animation::assignGfx( Animation* pA, const vector<BITMAP*>* pT )
{
  for (int i=0; i < pA->frameCount(); i++) {
    int findex = pA->at(i).iFrameIndex;
    pA->at(i).setGraphics( EBitmap, pT->at( findex ) );
  }
}



/** Places the RLE-pointers to the frame structure.
 */
void Animation::assignRLE( Animation* pA, const vector<RLE_SPRITE*>* pT )
{
  for (int i=0; i < pA->frameCount(); i++) {
    int findex = pA->at(i).iFrameIndex;
    pA->at(i).setGraphics( ERleSprite, pT->at(findex) );    
  }
}


//********************************************************************
//                                                                   *
//      Constructors, destructor and operators                       *
//                                                                   *
//********************************************************************

/** Default constructor
 */
Animation::Animation() :
  iMode( MODE_LOOP ),
  iFrameDelayCounter( 0 ),
  iCurrentFrame( 0 ),
  iPaused( false ),
  iPlayDirection( 1 ),
  iSoundsOn( true )
{
}



/** A copy constructor
 */
Animation::Animation(const Animation& rO) :
  iMode( rO.iMode ),
  iFrameDelayCounter( rO.iFrameDelayCounter ),
  iCurrentFrame( rO.iCurrentFrame ),
  iPaused( rO.iPaused ),
  iFrameTable( rO.iFrameTable ),
  iPlayDirection( rO.iPlayDirection ),
  iSoundsOn( true )
{
}



/** Destructor
 */
Animation::~Animation()
{
}



/** operator =
 */
Animation& Animation::operator = (const Animation& rO)
{
  if (this != &rO) {
    iMode = rO.iMode;
    iFrameDelayCounter = rO.iFrameDelayCounter;
    iCurrentFrame = rO.iCurrentFrame;
    iPaused = rO.iPaused;
    iFrameTable = rO.iFrameTable;
    iPlayDirection = rO.iPlayDirection;
    iSoundsOn = rO.iSoundsOn;
  }
  return *this;
}


//********************************************************************
//                                                                   *
//      Public interface                                             *
//                                                                   *
//********************************************************************

/** Adds new frame
 */
void Animation::addFrame( const AnimFrame& f )
{
  iFrameTable.push_back( f );
}



/** Sets new framedata for index'th frame
 */
void Animation::setFrame( int index, const AnimFrame& f ) throw ( xOutOfBounds )
{
  try {
    iFrameTable.at( index ) = f;
  } catch ( std::exception& e ) {
    throw xOutOfBounds( "Animation", "setFrame(int index, const AnimFrame& f" );
  }
}



/** Deletes the index'th frame
 */
void Animation::delFrame( int index ) throw ( xOutOfBounds )
{
  if ( index < 0 || index >= this->frameCount() ) {
    throw xOutOfBounds( "Animation", "delFrame(int index)" );
  }
  vector<AnimFrame>::iterator it = iFrameTable.begin() + index;
  iFrameTable.erase( it );
}



/** Removes all the frames
 */
void Animation::clear()
{
  iFrameTable.clear();
  iCurrentFrame = 0;
  iFrameDelayCounter = 0;
  iMode = MODE_LOOP;
}



/** Sets the same delay for all the frames
 */
void Animation::setConstantDelay( int fdelay )
{
  for (int i=0; i < this->frameCount(); i++) {
    this->at(i).iFrameDelay = fdelay;
  }
}



/** Sets the playmode
 */
void Animation::setPlayMode( PlayMode m )
{
  iMode = m;
}



/** Jumps at the beginning of the animation
 */
void Animation::begin()
{
  iFrameDelayCounter = 0;
  iCurrentFrame = 0;
  iPlayDirection = 1;
}



/** Jumps to the index'th frame
 */
void Animation::seek( int index )
{
  iFrameDelayCounter = 0;
  iCurrentFrame = 0;
  iPlayDirection = 1;
}



/** Sets pausemode on or off
 */

void Animation::pause( bool p )
{
  iPaused = p;
}

/** Runs the animation. You should keep calling this method in regular
 * intervals.
 */
bool Animation::update()
{
  if ( this->paused() || iFrameTable.size() == 0 ) {
    return false;
  }
  
  bool ret = false;   // a framechange flag.
  iFrameDelayCounter++;
  
  if ( iMode == MODE_LOOP ) {
    // Check if current frame has been visible long enough.
    if ( iFrameDelayCounter > iFrameTable.at( iCurrentFrame ).iFrameDelay ) {

      // Go to next frame
      ret = true;
      iFrameDelayCounter = 0;
      iCurrentFrame += iPlayDirection;
    
      // Since playmode is loopped, we jump back to first frame once the
      // animation has been played through.
      if (iCurrentFrame >= this->frameCount() ) {
        iCurrentFrame = 0;
      
      }
    }
    
  } else if ( iMode == MODE_PINGPONG ) {
    if ( iFrameDelayCounter > iFrameTable.at( iCurrentFrame ).iFrameDelay ) {
      ret = true;
      iFrameDelayCounter = 0;
      iCurrentFrame += iPlayDirection;
      
      if (iCurrentFrame >= this->frameCount() ) {
        // End reached. Start played backwards.
        iCurrentFrame -= 2;
        iPlayDirection = -iPlayDirection;
        
      } else if (iCurrentFrame < 0) {
        // Begin reached. Start playing forward.
        iCurrentFrame = 1;
        iPlayDirection = -iPlayDirection;
      }
    }
    
  } else if (iMode == MODE_ONCE) {
    if (iFrameDelayCounter > iFrameTable.at(iCurrentFrame).iFrameDelay) {
      ret = true;
      iFrameDelayCounter = 0;
      iCurrentFrame++;
      // Check if the animation has been played through and stop it if
      // necessary.
      if (iCurrentFrame >= this->frameCount()) {
        iCurrentFrame--;
        this->pause( true );
      }
    }
    
  }
  
  // A framechange just occured. If the new frame has sound data, we
  // play the sample attached to this frame.
  if ( ret == true && iSoundsOn == true ) {
    int soundNum = iFrameTable.at( iCurrentFrame ).iFrameSound;
    if ( soundNum != -1 ) {
      Sound::playSample( soundNum, false );
    }
  }
  return ret;
}



/** Returns a reference to index'th frame
 */
AnimFrame& Animation::at(int index) throw ( xOutOfBounds )
{
  try {
    return iFrameTable.at(index);
    
  } catch ( std::exception& e ) {
    throw xOutOfBounds( "Animation", "at(int index)" );
  }
}



/** Returns the index'th frame
 */
AnimFrame Animation::at(int index) const throw ( xOutOfBounds )
{
  try {
    return iFrameTable.at(index);
    
  } catch ( std::exception& e ) {
    throw xOutOfBounds( "Animation", "at(int index) const" );
  }
}



/** Loads the animation from file.
 */
int Animation::load( const std::string& animfile )
{
  this->clear();
  std::ifstream fin( animfile.c_str() );
  if ( !fin ) {
    return KErrNotFound;
  }
  
  std::string tmp;
  while ( true ) {
    if ( fin.eof() == true ) {
      fin.close();
      return KErrEof;
    }
    fin >> tmp;
    if ( tmp == KStrCommentLine ) {
      fin.ignore(4096, KCharNewLine);
      
    } else if ( tmp == ANIMATION_OPENINGTAG ) {
      int err = this->read( fin );
      if ( err != KErrNone ) {
        fin.close();
        return err;
      }
      break;
      
    }
  }
  fin.close();
  return KErrNone;
}



/** Reads the animation data from given file.
 */
int Animation::read( std::istream& rFin )
{
  this->clear();
  
  // At first there should be the <animation>-tag
  std::string tmp;
  while ( true ) {
    if ( rFin.eof() ) {
      return KErrEof;
    }
    
    rFin >> tmp;
    if ( tmp == ANIMATION_CLOSINGTAG ) {
      // End tag encountered. The animation has been read.
      break;
      
    } else if ( tmp == HEADER_OPENINGTAG ) {
      int ret = this->readHeaderElement( rFin );
      if ( ret != KErrNone ) {
        return ret;
      }
      
    } else if ( tmp == BODY_OPENINGTAG ) {
      int ret = this->readBodyElement( rFin );
      if ( ret != KErrNone ) {
        return ret;
      }
      
    } else if ( tmp == KStrCommentLine ) {
      // Comment line
      rFin.ignore(4096, KCharNewLine);
      
    } else {
      // Unsupported tag. We pass it.
      
    }
  }
  return KErrNone;
}



/** Sets the soundmode
 */
void Animation::setSoundMode( bool aSoundsOn )
{
  iSoundsOn = aSoundsOn;
}



//********************************************************************
//                                                                   *
//      Public GET - methods                                         *
//                                                                   *
//********************************************************************

/** Returns the index of the current frame
 */
int Animation::getPlayPosition() const
{
  return iCurrentFrame;
}



/** Tells, if pause is on
 */
bool Animation::paused() const
{
  return iPaused;
}



/** Returns the playmode
 */
Animation::PlayMode Animation::getPlayMode() const
{
  return iMode;
}



/** Returns the current frame
 */
AnimFrame Animation::currentFrame() const throw ( xOutOfBounds )
{
  try {
    return iFrameTable.at( iCurrentFrame );
    
  } catch ( std::exception& e ) {
    throw xOutOfBounds( "Animation", "currentFrame()" );
    
  }
}



/** Returns the frame count
 */
int Animation::frameCount() const
{
  return iFrameTable.size();
}



/** Returns the sounds flag.
 */
bool Animation::soundsOn() const
{
  return iSoundsOn;
}



/** Returns the uid of this animation.
 */
int Animation::uid() const
{
  return iUid;
}



/** Returns the uid of the GfxObject
 */
int Animation::gfxUid() const
{
  return iGfxUid;
}



//********************************************************************
//                                                                   *
//      Private methods                                              *
//                                                                   *
//********************************************************************

/** Reads the header information.
 */
int Animation::readHeaderElement( istream& aIn )
{
  while ( aIn.eof() == false ) {
    string tmp;
    aIn >> tmp;
    if ( tmp == KStrCommentLine ) {
      aIn.ignore(4096, KCharNewLine);
      
    } else if ( tmp == HEADER_CLOSINGTAG ) {
      return KErrNone;
      
    } else if ( tmp == "playmode:" ) {
      if ( aIn.eof() ) return KErrEof;
      aIn >> tmp;
      if ( tmp == "once" ) {
        iMode = MODE_ONCE;
      } else if ( tmp == "loop" ) {
        iMode = MODE_LOOP;
      } else if ( tmp == "pingpong" ) {
        iMode = MODE_PINGPONG;
      } else {
        return KErrNotSupported;
      }
      
    } else if ( tmp == "uid:" ) {
      if ( aIn.eof() ) return KErrEof;
      aIn >> tmp;
      iUid = atoi( tmp.c_str() );
      
    } else if ( tmp == "gfx_uid:" ) {
      if ( aIn.eof() ) return KErrEof;
      aIn >> tmp;
      iGfxUid = atoi( tmp.c_str() );
      
    } else if ( tmp == "version:" ) {
      if ( aIn.eof() ) return KErrEof;
      aIn >> tmp;
      
    } else {
      return KErrNotSupported;
      
    }
  }
  return KErrEof;
}



/** Reads the BODY element.
 */
int Animation::readBodyElement( istream& aIn )
{
  while ( aIn.eof() == false ) {
    string tmp;
    aIn >> tmp;
    if ( tmp == BODY_CLOSINGTAG ) {
      return KErrNone;
      
    } else if ( tmp == KStrCommentLine ) {
      aIn.ignore(4096, KCharNewLine);
      
    } else if ( tmp == FRAME_OPENINGTAG ) {
      AnimFrame frame;
      int ret = frame.read(aIn);
      if ( ret != KErrNone ) {
        return ret;
      }
      this->addFrame( frame );
      
    } else {
      return KErrNotSupported;
      
    }
  }
  return KErrEof;
}


} // end of namespace
